'use strict';

/* --------------------------------------------------------------
 datatable_custom_sorting.js 2016-06-20
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Enable Custom DataTable Sorting
 *
 * DataTables will reset the table to the first page after sorting by default. As there is no way to override
 * this behavior, this module will remove the DataTable sorting event handlers and set its own, which will keep
 * the table to the current page. This module will also set a sort parameter to the URL on sorting change but will
 * not parse it during initialization. This must happen from the module that initializes the table.
 *
 * Important: This method will remove the click event from the "th.sorting" elements, so bind extra "click" events
 * after enabling the custom-sorting extension (on init.dt event).
 *
 * ### Events
 *
 * ```javascript
 * // Add custom callback once the column sorting was changed (the "info" object contains the column index,
 * // column name and sort direction: {index, name, direction}).
 * $('#datatable-instance').on('datatable_custom_sorting:change', function(event, info) {...});
 * ```
 *
 * @module Admin/Extensions/datatable_custom_sorting
 */
gx.extensions.module('datatable_custom_sorting', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * On Table Header Cell Click
     *
     * Perform the table sorting without changing the current page.
     */
    function _onTableHeaderCellClick() {
        // Change Table Order
        var index = $(this).index();
        var destination = $(this).hasClass('sorting_asc') ? 'desc' : 'asc';

        $this.DataTable().order([index, destination]).draw(false);

        // Trigger Event
        var order = $this.DataTable().order()[0];

        var _$this$DataTable$init = $this.DataTable().init(),
            columns = _$this$DataTable$init.columns;

        var info = {
            index: order[0],
            name: columns[order[0]].name,
            direction: order[1]
        };

        $this.trigger('datatable_custom_sorting:change', [info]);
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('preInit.dt', function () {
            $this.find('thead tr:first th.sorting').off('click').on('click', _onTableHeaderCellClick);
        });

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImRhdGF0YWJsZV9jdXN0b21fc29ydGluZy5qcyJdLCJuYW1lcyI6WyJneCIsImV4dGVuc2lvbnMiLCJtb2R1bGUiLCJkYXRhIiwiJHRoaXMiLCIkIiwiX29uVGFibGVIZWFkZXJDZWxsQ2xpY2siLCJpbmRleCIsImRlc3RpbmF0aW9uIiwiaGFzQ2xhc3MiLCJEYXRhVGFibGUiLCJvcmRlciIsImRyYXciLCJpbml0IiwiY29sdW1ucyIsImluZm8iLCJuYW1lIiwiZGlyZWN0aW9uIiwidHJpZ2dlciIsImRvbmUiLCJvbiIsImZpbmQiLCJvZmYiXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7Ozs7Ozs7QUFVQTs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBcUJBQSxHQUFHQyxVQUFILENBQWNDLE1BQWQsQ0FBcUIsMEJBQXJCLEVBQWlELEVBQWpELEVBQXFELFVBQVVDLElBQVYsRUFBZ0I7O0FBRWpFOztBQUVBO0FBQ0E7QUFDQTs7QUFFQTs7Ozs7O0FBS0EsUUFBTUMsUUFBUUMsRUFBRSxJQUFGLENBQWQ7O0FBRUE7Ozs7O0FBS0EsUUFBTUgsU0FBUyxFQUFmOztBQUVBO0FBQ0E7QUFDQTs7QUFFQTs7Ozs7QUFLQSxhQUFTSSx1QkFBVCxHQUFtQztBQUMvQjtBQUNBLFlBQU1DLFFBQVFGLEVBQUUsSUFBRixFQUFRRSxLQUFSLEVBQWQ7QUFDQSxZQUFNQyxjQUFjSCxFQUFFLElBQUYsRUFBUUksUUFBUixDQUFpQixhQUFqQixJQUFrQyxNQUFsQyxHQUEyQyxLQUEvRDs7QUFFQUwsY0FBTU0sU0FBTixHQUFrQkMsS0FBbEIsQ0FBd0IsQ0FBQ0osS0FBRCxFQUFRQyxXQUFSLENBQXhCLEVBQThDSSxJQUE5QyxDQUFtRCxLQUFuRDs7QUFFQTtBQUNBLFlBQU1ELFFBQVFQLE1BQU1NLFNBQU4sR0FBa0JDLEtBQWxCLEdBQTBCLENBQTFCLENBQWQ7O0FBUitCLG9DQVNiUCxNQUFNTSxTQUFOLEdBQWtCRyxJQUFsQixFQVRhO0FBQUEsWUFTeEJDLE9BVHdCLHlCQVN4QkEsT0FUd0I7O0FBVS9CLFlBQU1DLE9BQU87QUFDVFIsbUJBQU9JLE1BQU0sQ0FBTixDQURFO0FBRVRLLGtCQUFNRixRQUFRSCxNQUFNLENBQU4sQ0FBUixFQUFrQkssSUFGZjtBQUdUQyx1QkFBV04sTUFBTSxDQUFOO0FBSEYsU0FBYjs7QUFNQVAsY0FBTWMsT0FBTixDQUFjLGlDQUFkLEVBQWlELENBQUNILElBQUQsQ0FBakQ7QUFDSDs7QUFFRDtBQUNBO0FBQ0E7O0FBRUFiLFdBQU9XLElBQVAsR0FBYyxVQUFVTSxJQUFWLEVBQWdCO0FBQzFCZixjQUFNZ0IsRUFBTixDQUFTLFlBQVQsRUFBdUIsWUFBTTtBQUN6QmhCLGtCQUFNaUIsSUFBTixDQUFXLDJCQUFYLEVBQ0tDLEdBREwsQ0FDUyxPQURULEVBRUtGLEVBRkwsQ0FFUSxPQUZSLEVBRWlCZCx1QkFGakI7QUFHSCxTQUpEOztBQU1BYTtBQUNILEtBUkQ7O0FBVUEsV0FBT2pCLE1BQVA7QUFFSCxDQWxFRCIsImZpbGUiOiJkYXRhdGFibGVfY3VzdG9tX3NvcnRpbmcuanMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxyXG4gZGF0YXRhYmxlX2N1c3RvbV9zb3J0aW5nLmpzIDIwMTYtMDYtMjBcclxuIEdhbWJpbyBHbWJIXHJcbiBodHRwOi8vd3d3LmdhbWJpby5kZVxyXG4gQ29weXJpZ2h0IChjKSAyMDE2IEdhbWJpbyBHbWJIXHJcbiBSZWxlYXNlZCB1bmRlciB0aGUgR05VIEdlbmVyYWwgUHVibGljIExpY2Vuc2UgKFZlcnNpb24gMilcclxuIFtodHRwOi8vd3d3LmdudS5vcmcvbGljZW5zZXMvZ3BsLTIuMC5odG1sXVxyXG4gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cclxuICovXHJcblxyXG4vKipcclxuICogIyMgRW5hYmxlIEN1c3RvbSBEYXRhVGFibGUgU29ydGluZ1xyXG4gKlxyXG4gKiBEYXRhVGFibGVzIHdpbGwgcmVzZXQgdGhlIHRhYmxlIHRvIHRoZSBmaXJzdCBwYWdlIGFmdGVyIHNvcnRpbmcgYnkgZGVmYXVsdC4gQXMgdGhlcmUgaXMgbm8gd2F5IHRvIG92ZXJyaWRlXHJcbiAqIHRoaXMgYmVoYXZpb3IsIHRoaXMgbW9kdWxlIHdpbGwgcmVtb3ZlIHRoZSBEYXRhVGFibGUgc29ydGluZyBldmVudCBoYW5kbGVycyBhbmQgc2V0IGl0cyBvd24sIHdoaWNoIHdpbGwga2VlcFxyXG4gKiB0aGUgdGFibGUgdG8gdGhlIGN1cnJlbnQgcGFnZS4gVGhpcyBtb2R1bGUgd2lsbCBhbHNvIHNldCBhIHNvcnQgcGFyYW1ldGVyIHRvIHRoZSBVUkwgb24gc29ydGluZyBjaGFuZ2UgYnV0IHdpbGxcclxuICogbm90IHBhcnNlIGl0IGR1cmluZyBpbml0aWFsaXphdGlvbi4gVGhpcyBtdXN0IGhhcHBlbiBmcm9tIHRoZSBtb2R1bGUgdGhhdCBpbml0aWFsaXplcyB0aGUgdGFibGUuXHJcbiAqXHJcbiAqIEltcG9ydGFudDogVGhpcyBtZXRob2Qgd2lsbCByZW1vdmUgdGhlIGNsaWNrIGV2ZW50IGZyb20gdGhlIFwidGguc29ydGluZ1wiIGVsZW1lbnRzLCBzbyBiaW5kIGV4dHJhIFwiY2xpY2tcIiBldmVudHNcclxuICogYWZ0ZXIgZW5hYmxpbmcgdGhlIGN1c3RvbS1zb3J0aW5nIGV4dGVuc2lvbiAob24gaW5pdC5kdCBldmVudCkuXHJcbiAqXHJcbiAqICMjIyBFdmVudHNcclxuICpcclxuICogYGBgamF2YXNjcmlwdFxyXG4gKiAvLyBBZGQgY3VzdG9tIGNhbGxiYWNrIG9uY2UgdGhlIGNvbHVtbiBzb3J0aW5nIHdhcyBjaGFuZ2VkICh0aGUgXCJpbmZvXCIgb2JqZWN0IGNvbnRhaW5zIHRoZSBjb2x1bW4gaW5kZXgsXHJcbiAqIC8vIGNvbHVtbiBuYW1lIGFuZCBzb3J0IGRpcmVjdGlvbjoge2luZGV4LCBuYW1lLCBkaXJlY3Rpb259KS5cclxuICogJCgnI2RhdGF0YWJsZS1pbnN0YW5jZScpLm9uKCdkYXRhdGFibGVfY3VzdG9tX3NvcnRpbmc6Y2hhbmdlJywgZnVuY3Rpb24oZXZlbnQsIGluZm8pIHsuLi59KTtcclxuICogYGBgXHJcbiAqXHJcbiAqIEBtb2R1bGUgQWRtaW4vRXh0ZW5zaW9ucy9kYXRhdGFibGVfY3VzdG9tX3NvcnRpbmdcclxuICovXHJcbmd4LmV4dGVuc2lvbnMubW9kdWxlKCdkYXRhdGFibGVfY3VzdG9tX3NvcnRpbmcnLCBbXSwgZnVuY3Rpb24gKGRhdGEpIHtcclxuXHJcbiAgICAndXNlIHN0cmljdCc7XHJcblxyXG4gICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcbiAgICAvLyBWQVJJQUJMRVNcclxuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxyXG5cclxuICAgIC8qKlxyXG4gICAgICogTW9kdWxlIFNlbGVjdG9yXHJcbiAgICAgKlxyXG4gICAgICogQHR5cGUge2pRdWVyeX1cclxuICAgICAqL1xyXG4gICAgY29uc3QgJHRoaXMgPSAkKHRoaXMpO1xyXG5cclxuICAgIC8qKlxyXG4gICAgICogTW9kdWxlIEluc3RhbmNlXHJcbiAgICAgKlxyXG4gICAgICogQHR5cGUge09iamVjdH1cclxuICAgICAqL1xyXG4gICAgY29uc3QgbW9kdWxlID0ge307XHJcblxyXG4gICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcbiAgICAvLyBGVU5DVElPTlNcclxuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxyXG5cclxuICAgIC8qKlxyXG4gICAgICogT24gVGFibGUgSGVhZGVyIENlbGwgQ2xpY2tcclxuICAgICAqXHJcbiAgICAgKiBQZXJmb3JtIHRoZSB0YWJsZSBzb3J0aW5nIHdpdGhvdXQgY2hhbmdpbmcgdGhlIGN1cnJlbnQgcGFnZS5cclxuICAgICAqL1xyXG4gICAgZnVuY3Rpb24gX29uVGFibGVIZWFkZXJDZWxsQ2xpY2soKSB7XHJcbiAgICAgICAgLy8gQ2hhbmdlIFRhYmxlIE9yZGVyXHJcbiAgICAgICAgY29uc3QgaW5kZXggPSAkKHRoaXMpLmluZGV4KCk7XHJcbiAgICAgICAgY29uc3QgZGVzdGluYXRpb24gPSAkKHRoaXMpLmhhc0NsYXNzKCdzb3J0aW5nX2FzYycpID8gJ2Rlc2MnIDogJ2FzYyc7XHJcblxyXG4gICAgICAgICR0aGlzLkRhdGFUYWJsZSgpLm9yZGVyKFtpbmRleCwgZGVzdGluYXRpb25dKS5kcmF3KGZhbHNlKTtcclxuXHJcbiAgICAgICAgLy8gVHJpZ2dlciBFdmVudFxyXG4gICAgICAgIGNvbnN0IG9yZGVyID0gJHRoaXMuRGF0YVRhYmxlKCkub3JkZXIoKVswXTtcclxuICAgICAgICBjb25zdCB7Y29sdW1uc30gPSAkdGhpcy5EYXRhVGFibGUoKS5pbml0KCk7XHJcbiAgICAgICAgY29uc3QgaW5mbyA9IHtcclxuICAgICAgICAgICAgaW5kZXg6IG9yZGVyWzBdLFxyXG4gICAgICAgICAgICBuYW1lOiBjb2x1bW5zW29yZGVyWzBdXS5uYW1lLFxyXG4gICAgICAgICAgICBkaXJlY3Rpb246IG9yZGVyWzFdXHJcbiAgICAgICAgfTtcclxuXHJcbiAgICAgICAgJHRoaXMudHJpZ2dlcignZGF0YXRhYmxlX2N1c3RvbV9zb3J0aW5nOmNoYW5nZScsIFtpbmZvXSk7XHJcbiAgICB9XHJcblxyXG4gICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcbiAgICAvLyBJTklUSUFMSVpBVElPTlxyXG4gICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXHJcblxyXG4gICAgbW9kdWxlLmluaXQgPSBmdW5jdGlvbiAoZG9uZSkge1xyXG4gICAgICAgICR0aGlzLm9uKCdwcmVJbml0LmR0JywgKCkgPT4ge1xyXG4gICAgICAgICAgICAkdGhpcy5maW5kKCd0aGVhZCB0cjpmaXJzdCB0aC5zb3J0aW5nJylcclxuICAgICAgICAgICAgICAgIC5vZmYoJ2NsaWNrJylcclxuICAgICAgICAgICAgICAgIC5vbignY2xpY2snLCBfb25UYWJsZUhlYWRlckNlbGxDbGljayk7XHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIGRvbmUoKTtcclxuICAgIH07XHJcblxyXG4gICAgcmV0dXJuIG1vZHVsZTtcclxuXHJcbn0pOyBcclxuIl19
